/*
 * Decompiled with CFR 0.152.
 */
package net.sourceforge.peers.rtp;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.net.DatagramPacket;
import java.net.DatagramSocket;
import java.net.InetAddress;
import java.net.SocketException;
import java.net.SocketTimeoutException;
import java.security.AccessController;
import java.security.PrivilegedAction;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.RejectedExecutionException;
import net.sourceforge.peers.Logger;
import net.sourceforge.peers.rtp.RtpListener;
import net.sourceforge.peers.rtp.RtpPacket;
import net.sourceforge.peers.rtp.RtpParser;

public class RtpSession {
    private InetAddress remoteAddress;
    private int remotePort;
    private DatagramSocket datagramSocket;
    private ExecutorService executorService;
    private List<RtpListener> rtpListeners;
    private RtpParser rtpParser;
    private FileOutputStream rtpSessionOutput;
    private FileOutputStream rtpSessionInput;
    private boolean mediaDebug;
    private Logger logger;
    private String peersHome;

    public RtpSession(InetAddress localAddress, DatagramSocket datagramSocket, boolean mediaDebug, Logger logger, String peersHome) {
        this.mediaDebug = mediaDebug;
        this.logger = logger;
        this.peersHome = peersHome;
        this.datagramSocket = datagramSocket;
        this.rtpListeners = new ArrayList<RtpListener>();
        this.rtpParser = new RtpParser(logger);
        this.executorService = Executors.newSingleThreadExecutor();
    }

    public synchronized void start() {
        if (this.mediaDebug) {
            SimpleDateFormat simpleDateFormat = new SimpleDateFormat("yyyy-MM-dd_HH-mm-ss");
            String date = simpleDateFormat.format(new Date());
            String dir = this.peersHome + File.separator + "media" + File.separator;
            String fileName = dir + date + "_rtp_session.output";
            try {
                this.rtpSessionOutput = new FileOutputStream(fileName);
                fileName = dir + date + "_rtp_session.input";
                this.rtpSessionInput = new FileOutputStream(fileName);
            }
            catch (FileNotFoundException e) {
                this.logger.error("cannot create file", e);
                return;
            }
        }
        this.executorService.submit(new Receiver());
    }

    public void stop() {
        AccessController.doPrivileged(new PrivilegedAction<Void>(){

            @Override
            public Void run() {
                RtpSession.this.executorService.shutdown();
                return null;
            }
        });
    }

    public void addRtpListener(RtpListener rtpListener) {
        this.rtpListeners.add(rtpListener);
    }

    public synchronized void send(RtpPacket rtpPacket) {
        if (this.datagramSocket == null) {
            return;
        }
        byte[] buf = this.rtpParser.encode(rtpPacket);
        final DatagramPacket datagramPacket = new DatagramPacket(buf, buf.length, this.remoteAddress, this.remotePort);
        if (!this.datagramSocket.isClosed()) {
            AccessController.doPrivileged(new PrivilegedAction<Void>(){

                @Override
                public Void run() {
                    try {
                        RtpSession.this.datagramSocket.send(datagramPacket);
                    }
                    catch (IOException e) {
                        RtpSession.this.logger.error("cannot send rtp packet", e);
                    }
                    catch (SecurityException e) {
                        RtpSession.this.logger.error("security exception", e);
                    }
                    return null;
                }
            });
            if (this.mediaDebug) {
                try {
                    this.rtpSessionOutput.write(buf);
                }
                catch (IOException e) {
                    this.logger.error("cannot write to file", e);
                }
            }
        }
    }

    public void setRemoteAddress(InetAddress remoteAddress) {
        this.remoteAddress = remoteAddress;
    }

    public void setRemotePort(int remotePort) {
        this.remotePort = remotePort;
    }

    private void closeFileAndDatagramSocket() {
        if (this.mediaDebug) {
            try {
                this.rtpSessionOutput.close();
                this.rtpSessionInput.close();
            }
            catch (IOException e) {
                this.logger.error("cannot close file", e);
            }
        }
        AccessController.doPrivileged(new PrivilegedAction<Void>(){

            @Override
            public Void run() {
                RtpSession.this.datagramSocket.close();
                return null;
            }
        });
    }

    public boolean isSocketClosed() {
        if (this.datagramSocket == null) {
            return true;
        }
        return this.datagramSocket.isClosed();
    }

    class Receiver
    implements Runnable {
        Receiver() {
        }

        public void run() {
            int remotePort;
            int receiveBufferSize;
            try {
                receiveBufferSize = RtpSession.this.datagramSocket.getReceiveBufferSize();
            }
            catch (SocketException e) {
                RtpSession.this.logger.error("cannot get datagram socket receive buffer size", e);
                return;
            }
            byte[] buf = new byte[receiveBufferSize];
            final DatagramPacket datagramPacket = new DatagramPacket(buf, buf.length);
            boolean noException = false;
            boolean socketTimeoutException = true;
            int ioException = 2;
            int result = AccessController.doPrivileged(new PrivilegedAction<Integer>(){

                @Override
                public Integer run() {
                    try {
                        RtpSession.this.datagramSocket.receive(datagramPacket);
                    }
                    catch (SocketTimeoutException e) {
                        return 1;
                    }
                    catch (IOException e) {
                        RtpSession.this.logger.error("cannot receive packet", e);
                        return 2;
                    }
                    return 0;
                }
            });
            switch (result) {
                case 1: {
                    try {
                        RtpSession.this.executorService.execute(this);
                    }
                    catch (RejectedExecutionException rej) {
                        RtpSession.this.closeFileAndDatagramSocket();
                    }
                    return;
                }
                case 2: {
                    return;
                }
                case 0: {
                    break;
                }
            }
            InetAddress remoteAddress = datagramPacket.getAddress();
            if (remoteAddress != null && !remoteAddress.equals(RtpSession.this.remoteAddress)) {
                RtpSession.this.remoteAddress = remoteAddress;
            }
            if ((remotePort = datagramPacket.getPort()) != RtpSession.this.remotePort) {
                RtpSession.this.remotePort = remotePort;
            }
            byte[] data = datagramPacket.getData();
            int offset = datagramPacket.getOffset();
            int length = datagramPacket.getLength();
            byte[] trimmedData = new byte[length];
            System.arraycopy(data, offset, trimmedData, 0, length);
            if (RtpSession.this.mediaDebug) {
                try {
                    RtpSession.this.rtpSessionInput.write(trimmedData);
                }
                catch (IOException e) {
                    RtpSession.this.logger.error("cannot write to file", e);
                    return;
                }
            }
            RtpPacket rtpPacket = RtpSession.this.rtpParser.decode(trimmedData);
            for (RtpListener rtpListener : RtpSession.this.rtpListeners) {
                rtpListener.receivedRtpPacket(rtpPacket);
            }
            try {
                RtpSession.this.executorService.execute(this);
            }
            catch (RejectedExecutionException rej) {
                RtpSession.this.closeFileAndDatagramSocket();
            }
        }
    }
}

