package ru.infor.websocket.client;

import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.websocket.CloseReason;
import javax.websocket.CloseReason.CloseCodes;
import javax.websocket.MessageHandler;
import javax.websocket.OnClose;
import javax.websocket.OnMessage;
import javax.websocket.OnOpen;
import javax.websocket.Session;

import org.apache.log4j.Logger;

import ru.infor.websocket.transport.DataPack;
import ru.infor.websocket.transport.DataPackResponse;
import ru.infor.websocket.transport.SubscribingOptions;
import ru.infor.websocket.transport.SubscribingResult;
import ru.infor.websocket.transport.UnsubscribingOptions;
import ru.infor.websocket.transport.WSMessage;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.JsonDeserializationContext;
import com.google.gson.JsonDeserializer;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.JsonParseException;

@javax.websocket.ClientEndpoint
public final class ClientEndpoint {
	private static final Logger logger = Logger.getLogger(ClientEndpoint.class);

	Gson gson = new GsonBuilder().create();
	Session session;
	Thread t;

	Map<String, ClientHandler> subscribers = new LinkedHashMap<String, ClientHandler>();

	List<MessageHandler.Whole<String>> handlers = new LinkedList<MessageHandler.Whole<String>>();

	final Object closedLock = new Object();
	boolean closed = false;

	public void addHandler(MessageHandler.Whole<String> handler) {
		handlers.add(handler);
	}

	public boolean isClosed() {
		synchronized (closedLock) {
			return closed;
		}
	}

	@OnOpen
	public void onOpen(Session session) {
		this.session = session;
		session.setMaxTextMessageBufferSize(1024 * 1024); // 1Mb

		t = new Thread(new Runnable() {

			@Override
			public void run() {
				int counter = 0;
				while (true) {
					try {
						Thread.sleep(1000L);
					} catch (Exception e) {
					}

					try {
						ClientEndpoint.this.session.getBasicRemote().sendText("--PING--");
					} catch (Exception e) {
						ClientEndpoint.logger.error("Can't ping server: " + e.getMessage(), e);
						counter++;
						if (counter >= 5) {
							ClientEndpoint.this.onClose(ClientEndpoint.this.session, new CloseReason(
									CloseCodes.CLOSED_ABNORMALLY, "Connection lost"));
							break;
						}
					}
				}
			}
		});
		t.start();
	}

	@OnClose
	public void onClose(Session session, CloseReason closeReason) {
		logger.debug("session: " + session + " has been closed: " + closeReason);
		if (!this.session.equals(session))
			return;

		try {
			t.interrupt();
		} catch (Exception e) {

		}

		try {
			List<ClientHandler> handlers = new ArrayList<ClientHandler>();
			synchronized (subscribers) {
				handlers.addAll(subscribers.values());
			}
			if (handlers.isEmpty())
				return;

			Set<Object> set = new HashSet<Object>();
			for (ClientHandler h : handlers) {
				if (!SessionClosed.class.isAssignableFrom(h.instance.getClass()))
					continue;

				if (set.add(h.instance)) {
					SessionClosed s = (SessionClosed) h.instance;
					s.onClose();
				}
			}
		} finally {
			synchronized (closedLock) {
				closed = true;
			}
		}
	}

	@OnMessage
	public void onMessage(String message) {
		if ("--PONG--".equals(message)) {
			return;
		}

		WSMessageFake req = gson.fromJson(message, WSMessageFake.class);
		logger.debug("message: " + message);

		if (req.getMessageType().equals(SubscribingResult.class.getName()))
			for (MessageHandler.Whole<String> h : handlers)
				h.onMessage(message);

		final String key = getKey(req.getServiceName(), req.getMethodName(), req.getSid());
		ClientHandler handler = null;
		synchronized (subscribers) {
			handler = subscribers.get(key);
		}

		if (handler == null) {
			logger.info("No handler located by message: " + message);
			return;
		}

		if (!handler.isInit())
			logger.error("Handler: " + handler + " did not get valid subscription");

		if (handler.isInit() && req.getMessageType().equals(DataPack.class.getName())) {
			Gson gsonDP = new GsonBuilder().registerTypeAdapter(DataPack.class, new DataPackDeserializer(req)).create();
			DataPack pack = gsonDP.fromJson(message, DataPack.class);

			Object res = handler.invoke(pack.getClassName(), pack.getDataJson());
			if (res == null)
				return;

			DataPackResponse resp = new DataPackResponse(pack);

			boolean single = true;
			Class<?> cls = res.getClass();
			if (res.getClass().isArray()) {
				cls = res.getClass().getComponentType();
				single = false;
			} else if (Collection.class.isAssignableFrom(res.getClass())) {
				Collection<?> col = (Collection<?>) res;
				single = false;
				if (!col.isEmpty())
					cls = col.iterator().next().getClass();
			}

			resp.setClassName(cls.getName());
			resp.setDataJson((single) ? ("[" + gson.toJson(res) + "]") : gson.toJson(res));

			session.getAsyncRemote().sendText(gson.toJson(resp));
		}
	}

	public synchronized <T> String subscribe(final SubscribingOptions<T> options, ClientHandler handler) {
		SubscribeSIDHandler h = new SubscribeSIDHandler();
		try {
			try {
				this.handlers.add(h);
				this.session.getBasicRemote().sendText(gson.toJson(options));
			} catch (Exception e) {
				logger.error(e.getMessage(), e);
			}

			while (!h.isDone())
				try {
					Thread.sleep(100L);
				} catch (Exception e) {
				}

			if (h.getResult().getStatus() < 0)
				throw new RuntimeException("Subscribing error: " + h.getResult().getErrorMessage());

			if (h.getResult().getSid() != null) {
				String key = getKey(options.getServiceName(), options.getMethodName(), h.getResult().getSid());
				subscribers.put(key, handler);
				handler.setInit(true);
			}

			return h.getResult().getSid();
		} finally {
			this.handlers.remove(h);
		}
	}

	public static void main(String[] args) throws Exception {
		String message = "{\"msgId\":0,\"className\":\"ru.infor.ws.objects.vms.entities.NDData\",\"dataJson\":[{\"type\":{\"id\":1,\"isDeleted\":0},\"tripIndex\":18881,\"alarmDevice\":1,\"deviceId\":44862300,\"createdDateTime\":\"Jul 28, 2015 11:04:09 AM\",\"lat\":58.05233249,\"lon\":38.83423319,\"speed\":22.5,\"alarm\":0}],\"sid\":\"0014ed3a00526\",\"serviceName\":\"NDDataWS\",\"methodName\":\"sendList\",\"messageType\":\"ru.infor.websocket.transport.DataPack\"}";

		Gson gson = new GsonBuilder().create();
		WSMessageFake req = gson.fromJson(message, WSMessageFake.class);

		Gson gsonDP = new GsonBuilder().registerTypeAdapter(DataPack.class, new DataPackDeserializer(req)).create();
		DataPack pack = gsonDP.fromJson(message, DataPack.class);
		System.out.println(pack);
	}

	public static String getKey(String serviceName, String methodName, String id) {
		String key = serviceName + "/" + methodName + "/" + id;
		return key.toLowerCase();
	}

	public void unsubscribe(UnsubscribingOptions options) {
		String key = getKey(options.getServiceName(), options.getMethodName(), options.getSid());
		synchronized (subscribers) {
			subscribers.remove(key);
		}

		try {
			this.session.getBasicRemote().sendText(gson.toJson(options));
		} catch (Exception e) {
			logger.error(e.getMessage(), e);
		}
	}

}

class DataPackDeserializer implements JsonDeserializer<DataPack> {

	WSMessage msg;

	protected DataPackDeserializer(WSMessage msg) {
		super();
		this.msg = msg;
	}

	@Override
	public DataPack deserialize(JsonElement jElement, Type type, JsonDeserializationContext ctx)
			throws JsonParseException {
		JsonObject jObj = jElement.getAsJsonObject();
		final String clsName = jObj.get("className").getAsString();
		DataPack pack = new DataPack(msg);
		pack.setClassName(clsName);

		final Type t = (clsName != null) ? new ParameterizedType() {
			@Override
			public Type getRawType() {
				return java.util.List.class;
			}

			@Override
			public Type getOwnerType() {
				return null;
			}

			@Override
			public Type[] getActualTypeArguments() {
				try {
					return new Type[] { Class.forName(clsName) };
				} catch (Exception e) {
				}

				return new Type[] { HashMap.class };
			}
		} : java.util.List.class;

		pack.setDataJson((Collection<Object>) ctx.deserialize(jObj.get("dataJson"), t));
		pack.setMsgId(jObj.get("msgId").getAsInt());

		return pack;
	}

}

class WSMessageFake extends WSMessage {

}

class SubscribeSIDHandler implements MessageHandler.Whole<String> {

	private static final Logger logger = Logger.getLogger(SubscribeSIDHandler.class);

	boolean done = false;
	SubscribingResult result;
	Gson gson = new GsonBuilder().create();

	@Override
	public void onMessage(String msg) {
		SubscribingResult res = null;
		try {
			// WSMessageFake msgF = gson.fromJson(msg, WSMessageFake.class);
			// if
			// (!msgF.getMessageType().equals(SubscribingResult.class.getName()))
			// return;

			res = gson.fromJson(msg, SubscribingResult.class);
		} catch (Exception e) {
			logger.error(e.getMessage(), e);
		}

		setDone(true);
		this.result = res;
	}

	public SubscribingResult getResult() {
		return result;
	}

	public synchronized boolean isDone() {
		return done;
	}

	public synchronized void setDone(boolean done) {
		this.done = done;
	}
};