package ru.infor.websocket.server;

import java.lang.reflect.Array;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;

import javax.websocket.CloseReason;
import javax.websocket.Endpoint;
import javax.websocket.EndpointConfig;
import javax.websocket.MessageHandler;
import javax.websocket.Session;

import org.apache.log4j.Logger;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;

import ru.infor.json.JSONObject;
import ru.infor.websocket.transport.DataPack;
import ru.infor.websocket.transport.DataPackResponse;
import ru.infor.websocket.transport.SubscribingOptions;
import ru.infor.websocket.transport.SubscribingResult;
import ru.infor.websocket.transport.UnsubscribingOptions;
import ru.infor.websocket.transport.WSMessageWithContext;
import ru.infor.ws.business.ApplicationException;
import ru.infor.ws.core.MessageReceiverUtils;

public class SocketEndpoint extends Endpoint {

	protected static final Logger logger = Logger.getLogger(SocketEndpoint.class);

	protected static final Gson GSON = new GsonBuilder().create();

	protected static Map<String, Class<? extends SubscribingOptions<?>>> map = new HashMap<String, Class<? extends SubscribingOptions<?>>>();

	protected static Map<String, List<SubscriberSession>> subscribers = new LinkedHashMap<String, List<SubscriberSession>>();

	protected static AtomicInteger MSG_ID = new AtomicInteger(0);

	public static String getKey(String serviceName, String methodName) {
		String key = serviceName + "/" + methodName;
		return key.toLowerCase();
	}

	public static void registerSubscribingOptionsClass(String serviceName, String methodName,
			Class<? extends SubscribingOptions<?>> clazz) {
		map.put(getKey(serviceName, methodName), clazz);
	}

	public static <T> Map<String, Future<?>> invoke(String serviceName, String methodName, String sid,
			List<Object> values) {
		List<SubscriberSession> list = getSubscriberList(serviceName, methodName);
		if (list.isEmpty()) {
			logger.debug("No subscribers for: " + serviceName + "/" + methodName);
			return null;
		}

		Map<String, Future<?>> res = new HashMap<String, Future<?>>(list.size());
		for (SubscriberSession sSession : list) {
			if (sid != null && !sSession.options.getSid().equals(sid))
				continue;

			DataPack pack = new DataPack();
			pack.setMsgId(MSG_ID.addAndGet(1));
			if (!values.isEmpty())
				pack.setClassName(values.iterator().next().getClass().getName());
			pack.setDataJson(values);
			pack.setMethodName(methodName);
			pack.setServiceName(serviceName);
			pack.setSid(sSession.options.getSid()); // must be assigned during
													// subscription

			String message = GSON.toJson(pack);
			logger.debug("Message 2 send: " + message);

			try {
				synchronized (sSession) {
					if (sSession.session.isOpen()) {
						CallHandler<T> h = new CallHandler<T>(pack.getMsgId());
						res.put(sSession.options.getSid(), h);
						((BrokerMessageHandler) sSession.session.getMessageHandlers().iterator().next()).addHandler(h);
						sSession.session.getAsyncRemote().sendText(message);
					} else
						;
				}
			} catch (Throwable t) {
				logger.warn("Can't send socket notification: " + t.getMessage(), t);
			}
		}

		return res;
	}

	public static void closeAllConnections() {
		List<SubscriberSession> list = new ArrayList<SubscriberSession>();
		synchronized (subscribers) {
			for (List<SubscriberSession> l : subscribers.values()) {
				list.addAll(l);
			}

			subscribers.clear();
		}

		if (list.isEmpty())
			return;

		for (SubscriberSession sSession : list) {
			try {
				sSession.session.close();
			} catch (Throwable e) {
				logger.error(e.getMessage(), e);
			}
		}

	}

	public static List<SubscriberSession> getSubscriberList(String serviceName, String methodName) {
		List<SubscriberSession> list = new ArrayList<SubscriberSession>();
		synchronized (subscribers) {
			List<SubscriberSession> tmp = subscribers.get(getKey(serviceName, methodName));
			if (tmp != null)
				list.addAll(tmp);
		}
		return list;
	}

	public static void notify(String serviceName, String methodName, String sid, List<Object> values) {
		List<SubscriberSession> list = getSubscriberList(serviceName, methodName);
		if (list.isEmpty()) {
			logger.debug("No subscribers for: " + serviceName + "/" + methodName);
			return;
		}

		for (SubscriberSession sSession : list) {
			if (sid != null && !sid.equals(sSession.options.getSid()))
				continue;

			Collection<Object> values2Send = sSession.options.filter(values);
			if (values2Send.isEmpty()) {
				logger.debug("No data to send to: " + sSession.session);
				continue;
			}

			DataPack pack = new DataPack();
			pack.setClassName(values2Send.iterator().next().getClass().getName());
			pack.setDataJson(values2Send);
			pack.setMethodName(methodName);
			pack.setServiceName(serviceName);
			pack.setSid(sSession.options.getSid()); // must be assigned during
													// subscription

			String message = GSON.toJson(pack);
			logger.debug("Message 2 send: " + message);

			try {
				synchronized (sSession) {
					if (sSession.session.isOpen())
						sSession.session.getAsyncRemote().sendText(message);
					else
						;
				}
			} catch (Throwable t) {
				logger.warn("Can't send socket notification: " + t.getMessage(), t);
			}
		}
	}

	@Override
	public void onOpen(final Session session, EndpointConfig endpointConfig) {
		session.setMaxTextMessageBufferSize(1024 * 1024); // 1Mb
		session.addMessageHandler(new BrokerMessageHandler(session));
	}

	@Override
	public void onClose(Session session, CloseReason closeReason) {
		synchronized (subscribers) {
			logger.debug("Session closed: " + session + ". " + closeReason);
			int count = 0;
			for (List<SubscriberSession> list : subscribers.values()) {
				List<SubscriberSession> _2Remove = new LinkedList<SubscriberSession>();
				for (SubscriberSession s : list) {
					if (s.session.equals(session))
						_2Remove.add(s);
				}

				list.removeAll(_2Remove);
				count += _2Remove.size();

				for (SubscriberSession s : _2Remove)
					s.options.afterSessionClosed();
			}

			logger.debug(count + " subscribers were lost");
		}

		super.onClose(session, closeReason);
	}

	public static void main(String[] args) throws Exception {
		Map<String, Future<List<Object>>> obj = new HashMap<String, Future<List<Object>>>();

		System.out.println(obj.getClass());

		// String message =
		// "{\"serviceName\":\"NDDataWS\",\"methodName\":\"sendList\",\"messageType\":\"ru.infor.ws.business.vms.websocket.objects.SubscribingOptions_SendListNDData\",\"context\":{\"clientIPAddress\":\"109.111.79.6\",\"initiator\":\"public-transport-front-end\",\"userName\":\"test_alt\",\"password\":\"aY1RoZ2KEhzlgUmde3AWaA==\"},\"deviceIdList\":[9587,9593,9601,9633,9641,9649,9653,9201,87494905844,87494911675]}";
		// new BrokerMessageHandler(null).onMessage(message);
	}
}

final class FakeMessageWithContext extends WSMessageWithContext {
	private static final long serialVersionUID = -2681301205038326635L;
}

final class BrokerMessageHandler2 implements MessageHandler.Partial<String> {

	@Override
	public void onMessage(String arg0, boolean arg1) {
		// TODO Auto-generated method stub

	}
}

final class BrokerMessageHandler implements MessageHandler.Whole<String> {
	private static final Logger logger = Logger.getLogger(BrokerMessageHandler.class);
	protected static final Gson GSON = new GsonBuilder().create();
	protected static AtomicLong SID = new AtomicLong(System.currentTimeMillis());

	private Session session;

	private List<MessageHandler.Whole<String>> handlers = new LinkedList<MessageHandler.Whole<String>>();

	protected BrokerMessageHandler(Session session) {
		super();

		this.session = session;
	}

	public synchronized void addHandler(MessageHandler.Whole<String> h) {
		if (handlers.size() < 10) {
			handlers.add(h);
			return;
		}

		List<MessageHandler.Whole<String>> tmp = new LinkedList<MessageHandler.Whole<String>>();
		tmp.add(h);
		for (Whole<String> h1 : handlers) {
			if (h1 instanceof CallHandler<?>) {
				if (!((CallHandler<?>) h1).isDone())
					tmp.add(h1);
			}
		}
		handlers = tmp;
	}

	@Override
	public void onMessage(String meggase) {
		if ("--PING--".equals(meggase)) {
			try {
				session.getBasicRemote().sendText("--PONG--");
			} catch (Exception e) {
			}
			return; // nothing to do with ping message
		}

		try {
			JSONObject jObj = new JSONObject(meggase);
			final String msgType = jObj.getString("messageType");
			logger.info("message recieved: " + msgType);

			if (DataPackResponse.class.getName().equals(msgType)) {
				logger.debug("DataPackResponse message: " + meggase);

				List<MessageHandler.Whole<String>> tmp = new LinkedList<MessageHandler.Whole<String>>();
				synchronized (handlers) {
					tmp.addAll(handlers);
				}
				if (tmp.isEmpty())
					return;

				for (MessageHandler.Whole<String> h : tmp)
					h.onMessage(meggase);

				return;
			}

			logger.debug("Subscribing message: " + meggase);
			WSMessageWithContext msg = null;
			String key = null;
			try {
				msg = GSON.fromJson(meggase, FakeMessageWithContext.class);
				key = SocketEndpoint.getKey(msg.getServiceName(), msg.getMethodName());
				Class<? extends WSMessageWithContext> clazz = null;
				try {
					clazz = (Class<? extends WSMessageWithContext>) Class.forName(msgType);
				} catch (Throwable t) {
				}
				if (clazz != null)
					msg = GSON.fromJson(meggase, clazz);
			} catch (Exception e1) {
				logger.debug("Subscribing message deserialization error: " + e1.getMessage());
			}

			if (msg == null) {
				SubscribingResult result = SubscribingResult.getError("error: Invalid options");
				session.getBasicRemote().sendText(GSON.toJson(result));
				return;
			}

			if (msg.getContext() == null) {
				SubscribingResult result = SubscribingResult.getError("error: Invalid user context");
				result.setMethodName(msg.getMethodName());
				result.setServiceName(msg.getServiceName());
				session.getBasicRemote().sendText(GSON.toJson(result));
				return;
			}
			try {
				MessageReceiverUtils.checkUser(msg.getContext(), msg.getServiceName(), msg.getMethodName());
			} catch (Exception e1) {
				logger.error("Auth error: " + e1.getMessage());
				String errorCode = "user.not.found";
				if (e1 instanceof ApplicationException) {
					ApplicationException ae = (ApplicationException) e1;
					errorCode = ae.getFaultCode();
				}

				SubscribingResult result = SubscribingResult.getError("error: Invalid user context - " + errorCode);
				result.setMethodName(msg.getMethodName());
				result.setServiceName(msg.getServiceName());

				session.getBasicRemote().sendText(GSON.toJson(result));
				return;
			}

			if (msg instanceof UnsubscribingOptions) {
				synchronized (SocketEndpoint.subscribers) {
					List<SubscriberSession> sessions = SocketEndpoint.subscribers.get(key);
					if (sessions != null) {
						SubscriberSession sSession = null;
						for (SubscriberSession s : sessions) {
							if (s.options.getSid().equals(msg.getSid())) {
								sSession = s;
								break;
							}
						}

						if (sSession != null) {
							sessions.remove(sSession);
							if (sessions.isEmpty())
								SocketEndpoint.subscribers.remove(key);
						}
					}
				}
			} else {
				SubscribingResult result = SubscribingResult.getSuccess();
				result.setMethodName(msg.getMethodName());
				result.setServiceName(msg.getServiceName());

				if (msg.getSid() == null)
					msg.setSid(generateSID());

				result.setSid(msg.getSid());
				session.getBasicRemote().sendText(GSON.toJson(result));

				synchronized (SocketEndpoint.subscribers) {
					List<SubscriberSession> sessions = SocketEndpoint.subscribers.get(key);
					if (sessions == null) {
						sessions = new LinkedList<SubscriberSession>();
						SocketEndpoint.subscribers.put(key, sessions);
					}

					SubscriberSession sSession = null;
					for (SubscriberSession s : sessions) {
						if (s.options.getSid().equals(msg.getSid())) {
							sSession = s;
							break;
						}
					}

					if (sSession != null) {
						try {
							sSession.options.afterSessionClosed();
						} catch (Throwable e) {
							logger.error("afterSessionClosed error: " + e.getMessage(), e);
						}
						sSession.options = (SubscribingOptions<Object>) msg;
					} else {
						sSession = new SubscriberSession();
						sSession.options = (SubscribingOptions<Object>) msg;
						sSession.session = session;

						sessions.add(sSession);
					}

					try {
						sSession.options.afterSubscriptionOpened();
					} catch (Throwable e) {
						logger.error("afterSubscriptionOpened error: " + e.getMessage(), e);
						throw new RuntimeException("invalid subscribing options ");
					}
				}

			}
		} catch (Exception e) {
			logger.error(e.getMessage(), e);
		}
	}

	private String generateSID() {
		long id = SID.getAndAdd(1L);
		return "00" + Long.toHexString(id);
	}
}

class CallHandler<T> implements MessageHandler.Whole<String>, Future<T> {

	private static final Logger logger = Logger.getLogger(CallHandler.class);

	int msgId;
	Object lock = new Object();
	Object val;
	boolean done = false;

	protected CallHandler(int msgId) {
		super();
		this.msgId = msgId;
	}

	@Override
	public void onMessage(String message) {
		synchronized (lock) {
			if (done)
				return;
		}

		JSONObject jObj = new JSONObject(message);
		int id = -1;
		try {
			id = jObj.getInt("replyToMsgId");
		} catch (Exception e) {
		}
		if (id != msgId)
			return;

		final DataPackResponse resp = SocketEndpoint.GSON.fromJson(message, DataPackResponse.class);

		if (resp.getClassName() != null)
			try {
				boolean isArray = resp.getDataJson().startsWith("[");
				Class<?> cls = Class.forName(resp.getClassName());
				if (isArray)
					cls = Array.newInstance(cls, 0).getClass();

				Object mtdResult = SocketEndpoint.GSON.fromJson(resp.getDataJson(), cls);
				logger.debug("data: " + mtdResult);
				synchronized (lock) {
					val = (isArray) ? Arrays.asList((Object[]) mtdResult) : Arrays.asList(mtdResult);
				}
			} catch (Exception e) {
			} finally {
				synchronized (lock) {
					done = true;
				}
			}
	}

	@Override
	public boolean cancel(boolean mayInterruptIfRunning) {
		return false;
	}

	@Override
	public boolean isCancelled() {
		return false;
	}

	@Override
	public boolean isDone() {
		synchronized (lock) {
			return done;
		}
	}

	@Override
	public T get() throws InterruptedException, ExecutionException {
		synchronized (lock) {
			return (T) val;
		}
	}

	@Override
	public T get(long timeout, TimeUnit unit) throws InterruptedException, ExecutionException, TimeoutException {
		return get();
	}

}